'use strict';

var _typeof = typeof Symbol === "function" && typeof Symbol.iterator === "symbol" ? function (obj) { return typeof obj; } : function (obj) { return obj && typeof Symbol === "function" && obj.constructor === Symbol && obj !== Symbol.prototype ? "symbol" : typeof obj; };

/* --------------------------------------------------------------
 cart_handler.js 2025-08-19
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 2025 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 --------------------------------------------------------------
 */

/**
 * Component for handling the add to cart and wishlist features
 * at the product details and the category listing pages. It cares
 * for attributes, properties, quantity and all other
 * relevant data for adding an item to the basket or wishlist
 */
gambio.widgets.module('cart_handler', ['hooks', 'form', 'xhr', 'loading_spinner', gambio.source + '/libs/events', gambio.source + '/libs/modal.ext-magnific', gambio.source + '/libs/modal'], function (data) {

	'use strict';

	// ########## VARIABLE INITIALIZATION ##########

	var $this = $(this),
	    $body = $('body'),
	    $window = $(window),
	    busy = false,
	    ajax = null,
	    timeout = 0,
	    previousModifiers = {},
	    defaults = {
		// AJAX "add to cart" URL
		addCartUrl: 'shop.php?do=Cart/BuyProduct',
		// AJAX "add to cart" URL for customizer products
		addCartCustomizerUrl: 'shop.php?do=Cart/Add',
		// AJAX URL to perform a value check
		checkUrl: 'shop.php?do=CheckStatus',
		// AJAX URL to perform the add to wishlist
		wishlistUrl: 'shop.php?do=WishList/Add',
		// Submit URL for price offer button
		priceOfferUrl: 'gm_price_offer.php',
		// Submit method for price offer
		priceOfferMethod: 'get',
		// Selector for the cart dropdown
		dropdown: '#head_shopping_cart',
		// "Add to cart" buttons selectors
		cartButtons: '.js-btn-add-to-cart',
		// "Wishlist" buttons selectors
		wishlistButtons: '.btn-wishlist',
		// "Price offer" buttons selectors
		priceOfferButtons: '.btn-price-offer',
		// Selector for the attribute fields
		attributes: '.js-calculate',
		// Selector for product property
		productOptions: '.modifier-group .modifier-content .modifier-item',
		productOptionField: '.hidden-input',
		// Selector for the quantity
		quantity: '.js-calculate-qty',
		// URL where to get the theme for the dropdown
		tpl: null,
		// Show attribute images in product images swiper (if possible)
		// -- this feature is not supported yet --
		attributImagesSwiper: false,
		// Trigger the attribute images to this selectors
		triggerAttrImagesTo: '#product_image_swiper, #product_thumbnail_swiper, ' + '#product_thumbnail_swiper_mobile',
		// Class that gets added to the button on processing
		processingClass: 'loading',
		// Duration for that the success or fail class gets added to the button
		processingDuration: 2000,
		// AJAX response content selectors
		selectorMapping: {
			buttons: '.shopping-cart-button',
			giftContent: '.gift-cart-content-wrapper',
			giftLayer: '.gift-cart-layer',
			shareContent: '.share-cart-content-wrapper',
			shareLayer: '.share-cart-layer',
			hiddenOptions: '#cart_quantity .hidden-options',
			message: '.global-error-messages',
			messageCart: '.cart-error-msg',
			messageHelp: '.help-block',
			modelNumber: '.model-number',
			modelNumberText: '.model-number-text',
			price: '.current-price-container',
			modifiersForm: '.modifiers-selection',
			quantity: '.products-quantity-value',
			quantityInfo: '.products-quantity',
			ribbonSpecial: '.ribbon-special',
			shippingInformation: '#shipping-information-layer',
			shippingTime: '.products-shipping-time-value',
			shippingTimeImage: '.img-shipping-time img',
			totals: '#cart_quantity .total-box',
			weight: '.products-details-weight-container span',
			abroadShippingInfo: '.abroad-shipping-info'
		},
		page: 'product-listing'
	},
	    options = $.extend(true, {}, defaults, data),
	    module = {},
	    mobile = $(window).width() <= 767;

	// ########## HELPER FUNCTIONS ##########

	/**
  * Helper function that updates the button
  * state with an error or success class for
  * a specified duration
  * @param   {object}        $target         jQuery selection of the target button
  * @param   {string}        state           The state string that gets added to the loading class
  * @private
  */
	var _addButtonState = function _addButtonState($target, state) {
		var timer = setTimeout(function () {
			$target.removeClass(options.processingClass + ' ' + options.processingClass + state);
		}, options.processingDuration);

		$target.data('timer', timer).addClass(options.processingClass + state);
	};

	/**
  * Helper function to set the messages and the
  * button state.
  * @param       {object}    data                Result form the ajax request
  * @param       {object}    $form               jQuery selecion of the form
  * @param       {boolean}   disableButtons      If true, the button state gets set to (in)active
  * @param       {boolean}   showNoCombiMesssage If true, the error message for missing property combination selection will be displayed
  * @private
  */
	var _stateManager = function _stateManager(data, $form, disableButtons, showNoCombiSelectedMesssage) {

		// Save the current changed modifier
		var lastFocusedElement = $(':focus');
		var lastFocusedElementID = void 0;
		if (lastFocusedElement.hasClass('modifier-item')) {
			lastFocusedElementID = '[data-value="' + lastFocusedElement.data('value') + '"]';
		} else {
			var lastFocusedElementName = lastFocusedElement.attr('name');
			var isLastFocusedElementRadio = lastFocusedElement.attr('type') === 'radio';
			lastFocusedElementID = isLastFocusedElementRadio ? '[name="' + lastFocusedElementName + '"]:checked' : '[name="' + lastFocusedElementName + '"]';
		}

		// Remove the attribute images from the common content
		// so that it doesn't get rendered anymore. Then trigger
		// an event to the given selectors and deliver the
		// attrImages object
		if (options.attributImagesSwiper && data.attrImages && data.attrImages.length) {
			delete data.content.images;
			$(options.triggerAttrImagesTo).trigger(jse.libs.theme.events.SLIDES_UPDATE(), { attributes: data.attrImages });
		}

		// Set the messages given inside the data.content object
		$.each(data.content, function (i, v) {
			var $element = $body.hasClass('page-product-info') ? $this.find(options.selectorMapping[v.selector]) : $form.parent().find(options.selectorMapping[v.selector]);

			if ((!showNoCombiSelectedMesssage || v.value === '') && i === 'messageNoCombiSelected') {
				return true;
			}

			switch (v.type) {
				case 'hide':
					if (v.value === 'true') {
						$element.hide();
					} else {
						$element.show();
					}
					break;
				case 'html':
					$element.html(v.value);
					break;
				case 'attribute':
					$element.attr(v.key, v.value);
					break;
				case 'replace':
					if (v.value) {
						$element.replaceWith(v.value);
					} else {
						$element.addClass('hidden').empty();
					}
					break;
				default:
					$element.text(v.value);
					break;
			}
		});

		// Dis- / Enable the buttons
		if (disableButtons) {
			var $buttons = $form.find(options.cartButtons);
			if (data.success) {
				$buttons.removeClass('inactive');
				$buttons.removeClass('btn-inactive');
				$buttons.prop("disabled", false);
			} else {
				$buttons.addClass('inactive');
				$buttons.addClass('btn-inactive');
				$buttons.prop("disabled", true);
			}
		}

		if (data.content.message) {
			var $errorField = $form.find(options.selectorMapping[data.content.message.selector]);
			if (data.content.message.value) {
				$errorField.removeClass('hidden').show();
			} else {
				$errorField.addClass('hidden').hide();

				if (showNoCombiSelectedMesssage && data.content.messageNoCombiSelected !== undefined && data.content.messageNoCombiSelected) {
					if (data.content.messageNoCombiSelected.value) {
						$errorField.removeClass('hidden').show();
					} else {
						$errorField.addClass('hidden').hide();
					}
				}
			}
		}

		$window.trigger(jse.libs.theme.events.STICKYBOX_CONTENT_CHANGE());

		// Focus the last changed modifier
		if ($body.hasClass('page-product-info')) {
			$(document).find(lastFocusedElementID).first().focus();
		}
	};

	/**
  * Helper function to send the ajax
  * On success redirect to a given url, open a layer with
  * a message or add the item to the cart-dropdown directly
  * (by triggering an event to the body)
  * @param       {object}      data      Form data
  * @param       {object}      $form     The form to fill
  * @param       {string}      url       The URL for the AJAX request
  * @private
  */
	var _addToSomewhere = function _addToSomewhere(data, $form, url, $button) {
		function callback() {
			jse.libs.xhr.post({ url: url, data: data }, true).done(function (result) {
				try {
					// Fill the page with the result from the ajax
					_stateManager(result, $form, false);

					// If the AJAX was successful execute
					// a custom functionality
					if (result.success) {
						switch (result.type) {
							case 'url':
								if (result.url.substr(0, 4) !== 'http') {
									location.href = jse.core.config.get('appUrl') + '/' + result.url;
								} else {
									location.href = result.url;
								}

								break;
							case 'dropdown':
								$body.trigger(jse.libs.theme.events.CART_UPDATE(), [true]);
								break;
							case 'layer':
								jse.libs.theme.modal.info({ title: result.title, content: result.msg });
								break;
							default:
								break;
						}
					}
				} catch (ignore) {}
				_addButtonState($button, '-success');
			}).fail(function () {
				_addButtonState($button, '-fail');
			}).always(function () {
				// Reset the busy flag to be able to perform
				// further AJAX requests
				busy = false;
			});
		}

		if (!busy) {
			// only execute the ajax
			// if there is no pending ajax call
			busy = true;

			jse.libs.hooks.execute(jse.libs.hooks.keys.shop.cart.add, data, 500).then(callback).catch(callback);
		}
	};

	// ########## EVENT HANDLER ##########

	/**
  * Hide the busy/loading-animation on the "add to cart" button.
  *
  * @param $form
  * @private
  */
	var _removeAddToCartThrobbler = function _removeAddToCartThrobbler($form) {
		var $btn = $this.find("[name=btn-add-to-cart]");
		var $btnFake = $this.find(".btn-add-to-cart-fake");
		if ($btnFake) {
			$btnFake.hide().prop("disabled", false).find('.throbbler').remove();
		}
		if ($btn) {
			$btn.show();
		}
	};

	/**
 * Handler for the submit form / click
 * on "add to cart" & "wishlist" button.
 * It performs a check on the availability
 * of the combination and quantity. If
 * successful it performs the add to cart
 * or wishlist action, if it's not a
 * "check" call
 * @param       {object}    e      jQuery event object
 * @private
 */
	var _submitHandler = function _submitHandler(e) {
		if (e) {
			e.preventDefault();
		}

		var $self = $(this),
		    $form = $self.is('form') ? $self : $self.closest('form'),
		    customizer = $form.hasClass('customizer'),
		    properties = !!$form.find('.properties-selection-form').length,
		    module = properties ? '' : '/Attributes',
		    showNoCombiSelectedMesssage = e && e.data && e.data.target && e.data.target !== 'check';

		if ($form.length) {

			// Show properties overlay
			// to disable user interaction
			// before markup replace
			if (properties) {
				$this.addClass('loading');
			}

			if ($self.is('select')) {
				var price = $self.find(":selected").attr('data-price');
				$self.parents('.modifier-group').find('.selected-value-price').text(price);
			}

			var getGalleryHash = $('#current-gallery-hash').val();
			$form.find('#update-gallery-hash').val(getGalleryHash);

			var formdata = jse.libs.form.getData($form, null, true);
			formdata.target = e && e.data && e.data.target ? e.data.target : 'check';
			formdata.isProductInfo = $form.hasClass('product-info') ? 1 : 0;

			// Abort previous check ajax if
			// there is one in progress
			if (ajax && e) {
				ajax.abort();
			}

			// Add processing-class to the button
			// and remove old timed events
			if (formdata.target !== 'check') {
				var timer = $self.data('timer');
				if (timer) {
					clearTimeout(timer);
				}

				$self.removeClass(options.processingClass + '-success ' + options.processingClass + '-fail').addClass(options.processingClass);
			}

			formdata.previousModifiers = previousModifiers;

			ajax = jse.libs.xhr.get({
				url: options.checkUrl + module,
				data: formdata
			}, true).done(function (result) {
				_stateManager(result, $form, true, showNoCombiSelectedMesssage);
				$this.removeClass('loading');

				// Check if the gallery images changed
				if (formdata.target === 'check' && result.content.imageGallery.trim() !== '' && result.content.replaceGallery === true && formdata.isProductInfo === 1) {
					var loadingSpinner = jse.libs.loading_spinner.show($('.product-info-stage'), 9999);

					var swipers = [$('#product_image_swiper'), $('#product_thumbnail_swiper'), $('#product_thumbnail_swiper_mobile')];

					var _iteratorNormalCompletion = true;
					var _didIteratorError = false;
					var _iteratorError = undefined;

					try {
						for (var _iterator = swipers[Symbol.iterator](), _step; !(_iteratorNormalCompletion = (_step = _iterator.next()).done); _iteratorNormalCompletion = true) {
							var element = _step.value;

							var instance = element.swiper();

							if ((typeof instance === 'undefined' ? 'undefined' : _typeof(instance)) !== 'object') {
								continue;
							}

							instance.destroy(true, true);
							element.off().remove();
						}
					} catch (err) {
						_didIteratorError = true;
						_iteratorError = err;
					} finally {
						try {
							if (!_iteratorNormalCompletion && _iterator.return) {
								_iterator.return();
							}
						} finally {
							if (_didIteratorError) {
								throw _iteratorError;
							}
						}
					}

					$('#image-collection-container').html(result.content.imageGallery);
					$('#product_image_layer').html(result.content.imageModal);

					gambio.widgets.init($('.product-info-content')).done(function () {
						jse.libs.loading_spinner.hide(loadingSpinner);
					});
				} else if (formdata.target === 'check' && result.content.imageGallery.trim() === '' && result.content.replaceGallery === true) {
					$('#image-collection-container').html(result.content.imageGallery);
					$('#product_image_layer').html(result.content.imageModal);
				}

				if (result.success) {
					var event = null,
					    url = null;

					switch (formdata.target) {
						case 'wishlist':
							if (customizer) {
								event = jse.libs.theme.events.ADD_CUSTOMIZER_WISHLIST();
							}
							url = options.wishlistUrl;
							break;
						case 'cart':
							if (customizer) {
								event = jse.libs.theme.events.ADD_CUSTOMIZER_CART();
								url = options.addCartCustomizerUrl;
							} else {
								url = options.addCartUrl;
							}
							break;
						case 'price_offer':
							$form.attr('action', options.priceOfferUrl).attr('method', options.priceOfferMethod);
							$form.off('submit');
							$form.submit();

							return;
						default:
							setTimeout(function () {
								$window.trigger(jse.libs.theme.events.STICKYBOX_CONTENT_CHANGE());
							}, 250);
							break;
					}

					if (event) {
						var deferred = $.Deferred();
						deferred.done(function (customizerRandom) {
							formdata[customizerRandom] = 0;
							_addToSomewhere(formdata, $form, url, $self);
						}).fail(function () {
							_addButtonState($self, '-fail');
						});
						$body.trigger(event, [{ 'deferred': deferred, 'dataset': formdata }]);
					} else if (url) {
						_addToSomewhere(formdata, $form, url, $self);
					}
				} else {
					_removeAddToCartThrobbler($form);
				}

				if (formdata.target === 'check') {
					previousModifiers = formdata.modifiers;
				}
			}).fail(function () {
				_removeAddToCartThrobbler($form);
				_addButtonState($self, '-fail');
			});
		}
	};

	/**
  * Handler for the change property option
  * */
	var _changeProductOptions = function _changeProductOptions(e) {
		var option = e.currentTarget;
		var optionValue = $(option).data('value');
		var optionContainer = $(option).parents('.modifier-group');

		$(optionContainer).find('li.active').removeClass('active');
		$(optionContainer).find('.modifier-item.active-modifier').removeClass('active-modifier');
		$(optionContainer).find('input.hidden-input').val(optionValue);
		$(optionContainer).find('input.hidden-input').trigger('blur', []);

		$(option).parents('li').addClass('active');
		$(option).addClass('active-modifier');
	};

	var _selectSelectedModifierInfo = function _selectSelectedModifierInfo(e) {
		var option = e.currentTarget;
		var price = $(option).attr('data-price');
		var label = $(option).attr('data-label');
		$(option).parents('.modifier-group').find('.selected-value-price').removeClass('temporary-value').attr('data-default-price', price);
		$(option).parents('.modifier-group').find('.selected-value').attr('data-default-value', label);
	};

	var _setSelectedModifierInfo = function _setSelectedModifierInfo(e) {
		var option = e.currentTarget;
		if (!$(option).parent().hasClass('active') && !$(option).is('select') && !$(option).hasClass('active-modifier')) {
			var price = $(option).attr('data-price');
			var label = $(option).attr('data-label');
			$(option).parents('.modifier-group').find('.selected-value-price').addClass('temporary-value').text(price);
			$(option).parents('.modifier-group').find('.selected-value').text(label);
		}
	};

	var _resetSelectedModifierInfo = function _resetSelectedModifierInfo(e) {
		var option = $(this);
		if (!$(option).parent().hasClass('active') && !$(option).is('select') && !$(option).hasClass('active-modifier')) {
			var priceHolder = $(option).parents('.modifier-group').find('.selected-value-price');
			var labelHolder = $(option).parents('.modifier-group').find('.selected-value');
			$(priceHolder).removeClass('temporary-value').text($(priceHolder).attr('data-default-price'));
			$(labelHolder).text($(labelHolder).attr('data-default-value'));
		}
	};

	/**
  * Keyup handler for quantity input field
  *
  * @param e
  * @private
  */
	var _keyupHandler = function _keyupHandler(e) {
		clearTimeout(timeout);

		timeout = setTimeout(function () {
			_submitHandler.call(this, e);
		}.bind(this), 300);
	};

	/**
  * Event handler for the add to cart button, that shows or hides the throbber.
  *
  * @param e
  */
	var _addToCartThrobberHandler = function _addToCartThrobberHandler(e) {
		var $btn = $this.find("[name=btn-add-to-cart]");
		var $btnFake = $this.find(".btn-add-to-cart-fake");
		var formReady = true;

		$(".properties-selection-form select").each(function () {
			var val = $(this).val();
			if (!val || val < 1) {
				formReady = false;
			}
		});

		if (formReady) {
			$btn.hide();
			$btnFake.show().prop("disabled", true).prepend('<span class="throbbler"></span>');
		}
	};

	/**
  * Cart dropdown oben event handler for the body.
  */
	var _cartDropdownOpenHandler = function _cartDropdownOpenHandler(e) {
		var $btn = $this.find("[name=btn-add-to-cart]");
		var $btnFake = $this.find(".btn-add-to-cart-fake");
		var fakeOrigLabel = $btnFake.html();
		var productCount = $(".cart-products-count").html();

		var textPhrases = JSON.parse($('#product-details-text-phrases').html());
		console.log(textPhrases['productsInCartSuffix']);

		$btnFake.html("<i class=\"fa fa-check\"></i> " + parseInt(productCount) + textPhrases['productsInCartSuffix']).prop("disabled", true).addClass("btn-buy-complete");

		setTimeout(function () {
			$btnFake.html(fakeOrigLabel).removeClass("btn-buy-complete").hide().prop("disabled", false);
			$(".throbbler", $btn).remove();
			$btn.show();
		}, 5000);
	};

	// ########## INITIALIZATION ##########

	/**
  * Init function of the widget
  * @constructor
  */
	module.init = function (done) {

		var $forms = $this.find('form');

		if (options.page === 'product-info') {
			$forms.on('submit', function (e) {
				_addToCartThrobberHandler(e);
			});
			$("body").on('CART_DROPDOWN_OPEN', _cartDropdownOpenHandler);
			$(document).on("mfpAfterClose", function (e) {
				_removeAddToCartThrobbler($forms);
			});
		}

		$forms.on('submit', { 'target': 'cart' }, _submitHandler).on('click', options.wishlistButtons, { 'target': 'wishlist' }, _submitHandler).on('click', options.priceOfferButtons, { 'target': 'price_offer' }, _submitHandler).on('change', options.attributes, { 'target': 'check' }, _submitHandler).on('mouseover', options.attributes, _setSelectedModifierInfo).on('mouseout', options.attributes, _resetSelectedModifierInfo).on('blur', options.productOptionField, { 'target': 'check' }, _submitHandler).on('click', options.productOptions, { 'target': 'check' }, function (e) {
			_selectSelectedModifierInfo(e);
			_changeProductOptions(e);
		}).on('mouseover', options.productOptions, _setSelectedModifierInfo).on('mouseout', options.productOptions, _resetSelectedModifierInfo).on('blur', options.quantity, { 'target': 'check' }, function (e) {
			_submitHandler(e);
		}).on('keyup', options.quantity, { 'target': 'check' }, _keyupHandler);

		// Fallback if the backend renders incorrect data
		// on initial page call
		$forms.not('.no-status-check').not('.product-info').each(function () {
			_submitHandler.call($(this));
		});
		done();
	};

	// Return data to widget engine
	return module;
});
//# sourceMappingURL=data:application/json;charset=utf8;base64,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
